# StyleSeat Web-Based Installer

## Overview

The StyleSeat application includes a comprehensive web-based installer that guides users through the complete setup process. This installer handles:

1. **System Requirements Checking** - Verifies PHP version and required extensions
2. **Database Configuration** - Guides database setup and optionally creates the database
3. **Application Configuration** - Sets up basic app settings
4. **Database Migration** - Creates all necessary database tables
5. **Database Seeding** - Optional: Populates test data
6. **Admin User Creation** - Creates the first administrator account
7. **Installation Finalization** - Locks the installer to prevent re-installation

## Accessing the Installer

When the application is first installed, visit:

```
http://localhost:8000/installer
```

The installer will automatically redirect here if not yet completed.

## Installation Steps

### Step 1: Welcome
- Overview of the installation process
- Prerequisites checklist
- Begin button to start installation

### Step 2: System Requirements
- Checks PHP version (8.2+)
- Verifies required extensions (PDO, OpenSSL, Mbstring, JSON)
- Confirms directory permissions (storage/, bootstrap/cache/)
- Verifies .env file exists and is writable

**Action**: Click "Next" to proceed if all checks pass

### Step 3: Database Configuration
- **Driver**: MySQL, PostgreSQL, SQLite, or SQL Server
- **Host**: Database server address (usually localhost)
- **Port**: Database port (3306 for MySQL)
- **Database**: Database name (default: styleseat)
- **Username**: Database user (usually root)
- **Password**: Database password (optional)
- **Create Database**: Automatically create the database if missing

**Actions**:
- Click "Test Connection" to verify database credentials
- Check "Create database" if you want the installer to create it
- Click "Next" to proceed

### Step 4: Application Configuration
- **App Name**: Display name (default: StyleSeat)
- **App URL**: Where your app will be accessible (http://localhost:8000)
- **Environment**: local, staging, or production
- **Debug Mode**: Automatically enabled for development

**Action**: Click "Next" to proceed

### Step 5: Database Migration
- Displays live migration output
- Creates all database tables (12 tables)
- **Seed Test Data**: Optional checkbox to populate demo data

**Actions**:
- Choose whether to seed test data
- Click "Run Migrations" to start
- Migrations should complete in seconds

### Step 6: Administrator User Creation
- **Name**: Admin full name
- **Email**: Login email (used for authentication)
- **Password**: Minimum 8 characters
- **Confirm Password**: Must match

**Action**: Click "Create Admin User" to proceed

### Step 7: Completion
- Displays success message
- Shows next steps
- Click "Complete Installation" to finalize

## Post-Installation

### File Created
After installation completes, the installer creates a flag file:
```
storage/installer_completed.flag
```

This file prevents the installer from running again.

### Login
Visit `http://localhost:8000` and log in with:
- **Email**: The administrator email you created
- **Password**: The administrator password you created

### Access Admin Panel
After login, navigate to `/admin/dashboard` to:
- Configure Firebase and Stripe
- Manage users
- View bookings and transactions
- Update global settings

## Re-installation

To run the installer again after completion:

1. Delete the installation flag file:
   ```
   rm storage/installer_completed.flag
   ```

2. Visit `/installer` again

⚠️ **Warning**: Re-running the installer will overwrite your .env file and may reset your application configuration.

## Features by Component

### InstallerController
Located at: `app/Http/Controllers/InstallerController.php`

**Key Methods**:
- `welcome()` - Display welcome page
- `requirements()` - Check system requirements
- `database()` - Database configuration form
- `testDatabase()` - Test database connection
- `saveDatabaseConfig()` - Save database settings to .env
- `configuration()` - Application configuration form
- `saveConfiguration()` - Save app settings
- `migration()` - Migration and seeding page
- `runMigrations()` - Execute migrations
- `runSeeders()` - Seed test data
- `admin()` - Admin user creation form
- `createAdmin()` - Create administrator user
- `completion()` - Installation completion page
- `finalize()` - Finalize installation and create flag

**Protected Methods**:
- `checkSystemRequirements()` - Verify system meets requirements
- `createDatabase()` - Create database via PDO
- `updateEnvFile()` - Modify .env file safely

### Middleware
Located at: `app/Http/Middleware/CheckInstallation.php`

Prevents access to application routes if installation is not complete. Routes starting with `/installer` are always accessible.

### Views
Located at: `resources/views/installer/`

- `layout.blade.php` - Main installer layout
- `welcome.blade.php` - Welcome page
- `requirements.blade.php` - System requirements
- `database.blade.php` - Database configuration
- `configuration.blade.php` - App configuration
- `migration.blade.php` - Migration and seeding
- `admin.blade.php` - Admin user creation
- `completion.blade.php` - Completion page

### Routes
Located at: `routes/web.php`

All installer routes use the `/installer` prefix:
```
GET    /installer                    - Welcome
GET    /installer/requirements       - Requirements
GET    /installer/database           - DB form
POST   /installer/database/test      - Test connection
POST   /installer/database/save      - Save config
GET    /installer/configuration      - App config form
POST   /installer/configuration/save - Save config
GET    /installer/migration          - Migration page
POST   /installer/migration/run      - Run migrations
POST   /installer/migration/seed     - Run seeders
GET    /installer/admin              - Admin form
POST   /installer/admin/create       - Create admin
GET    /installer/completion         - Completion
POST   /installer/finalize           - Finalize
```

## Error Handling

The installer gracefully handles errors:

### Database Connection Failed
- **Error**: "Connection failed: ..."
- **Solution**: Verify host, port, username, password
- **Action**: Click "Test Connection" again

### Database Creation Failed
- **Error**: "Could not create database: ..."
- **Solution**: Ensure user has CREATE privilege or create manually
- **Action**: Create database manually and leave "Create database" unchecked

### Migration Failed
- **Error**: Displayed in output box
- **Solution**: Check .env database configuration
- **Action**: Fix configuration and try again

### Permission Denied
- **Error**: "Could not write to .env file"
- **Solution**: Make .env file writable
- **Command**: `chmod 666 .env` (Linux/Mac)
- **Action**: Restart installer

### Admin User Already Exists
- **Error**: "Email already exists..."
- **Solution**: Use different email address
- **Action**: Change email and try again

## Configuration Options

### .env File Updates
The installer automatically updates these .env variables:

**Database**:
```
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=styleseat
DB_USERNAME=root
DB_PASSWORD=
```

**Application**:
```
APP_NAME=StyleSeat
APP_URL=http://localhost:8000
APP_ENV=local
APP_DEBUG=true
```

**Additional** (from seeding):
```
APP_KEY=base64:xxxxx (generated by Laravel)
```

### Manual Configuration
After installation, you can manually configure in `.env`:

**Firebase**:
```
FIREBASE_PROJECT_ID=your-project
FIREBASE_PRIVATE_KEY_ID=your-key-id
FIREBASE_PRIVATE_KEY="-----BEGIN PRIVATE KEY-----..."
```

**Stripe**:
```
STRIPE_PUBLIC_KEY=pk_test_xxx
STRIPE_SECRET_KEY=sk_test_xxx
```

**Mail**:
```
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=465
MAIL_USERNAME=your-username
MAIL_PASSWORD=your-password
```

## Security Considerations

### Installation Security
1. **Single-Use**: Once installation completes, installer is disabled
2. **CSRF Protection**: All forms use CSRF tokens
3. **Validation**: Input is validated on both client and server
4. **Error Logging**: Errors are logged for debugging

### Post-Installation
1. **Access Control**: Non-installer routes redirect to installer if not installed
2. **Environment Variables**: Sensitive data stored in .env, not in code
3. **Database Credentials**: Passed securely via AJAX
4. **File Permissions**: .env file should have restrictive permissions

### Recommendations
1. After installation:
   - Restrict .env file permissions: `chmod 600 .env`
   - Change database password
   - Update Stripe/Firebase credentials
   - Enable HTTPS in production

2. Before production:
   - Run `php artisan optimize`
   - Set `APP_DEBUG=false` in .env
   - Review all environment settings
   - Set strong admin password

## Troubleshooting

### Installer Not Showing
- **Problem**: Installer doesn't appear
- **Solution**: Check if `storage/installer_completed.flag` exists
- **Fix**: Delete flag file: `rm storage/installer_completed.flag`

### Can't Proceed Past Requirements
- **Problem**: Requirements show red X marks
- **Solution**: Install missing PHP extensions or update PHP
- **Example**: `php -m` to see installed extensions

### Database Connection Fails
- **Problem**: "Connection failed" error
- **Solution**: 
  - Verify MySQL is running
  - Check credentials
  - Ensure correct host/port
  - Try connecting from command line: `mysql -h host -u user -p`

### Migrations Don't Run
- **Problem**: "Migration failed" error
- **Solution**:
  - Check database connection
  - Verify database exists
  - Check user permissions
  - View error message for details

### Admin User Not Created
- **Problem**: "Email already exists" error
- **Solution**: Use a different email address

### After Installation
- **Problem**: Can't log in
- **Solution**: Verify email and password (case-sensitive)
- **Debug**: Check `storage/logs/laravel.log` for errors

## API Reference

### Database Testing
```javascript
POST /installer/database/test
{
  "driver": "mysql",
  "host": "localhost",
  "port": 3306,
  "database": "styleseat",
  "username": "root",
  "password": ""
}

Response:
{
  "success": true,
  "message": "Database connection successful!"
}
```

### Running Migrations
```javascript
POST /installer/migration/run

Response:
{
  "success": true,
  "message": "Migrations completed successfully!",
  "output": "[artisan output...]"
}
```

### Creating Admin
```javascript
POST /installer/admin/create
{
  "name": "John Doe",
  "email": "admin@example.com",
  "password": "secure-password",
  "password_confirmation": "secure-password"
}

Response:
{
  "success": true,
  "message": "Admin user created successfully!"
}
```

## Files Modified/Created

**New Files**:
- `app/Http/Controllers/InstallerController.php`
- `app/Http/Middleware/CheckInstallation.php`
- `resources/views/installer/layout.blade.php`
- `resources/views/installer/welcome.blade.php`
- `resources/views/installer/requirements.blade.php`
- `resources/views/installer/database.blade.php`
- `resources/views/installer/configuration.blade.php`
- `resources/views/installer/migration.blade.php`
- `resources/views/installer/admin.blade.php`
- `resources/views/installer/completion.blade.php`

**Modified Files**:
- `routes/web.php` - Added installer routes
- `app/Http/Kernel.php` - Added CheckInstallation middleware
- `app/Http/Middleware/*` - Created missing middleware files

## Support

For issues with the installer:
1. Check the troubleshooting section above
2. Review `storage/logs/laravel.log` for detailed errors
3. Verify system meets all requirements
4. Delete the installer flag and restart if needed

---

**StyleSeat v1.0 - Web-Based Installer**  
*One-click setup for beauty and wellness professionals*
