<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use Illuminate\View\View;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:admin');
    }

    public function index(): View
    {
        $bookings = Booking::with('client.user', 'professional.user', 'service')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.bookings.index', ['bookings' => $bookings]);
    }

    public function show($bookingId): View
    {
        $booking = Booking::with('client.user', 'professional.user', 'service', 'transaction', 'review')
            ->findOrFail($bookingId);

        return view('admin.bookings.show', ['booking' => $booking]);
    }

    public function filter(Request $request): View
    {
        $query = Booking::with('client.user', 'professional.user', 'service');

        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->filled('city')) {
            $query->where('city', $request->get('city'));
        }

        if ($request->filled('from_date')) {
            $query->whereDate('scheduled_at', '>=', $request->get('from_date'));
        }

        if ($request->filled('to_date')) {
            $query->whereDate('scheduled_at', '<=', $request->get('to_date'));
        }

        $bookings = $query->orderBy('scheduled_at', 'desc')->paginate(20);

        return view('admin.bookings.index', ['bookings' => $bookings]);
    }

    public function exportCsv()
    {
        $bookings = Booking::with('client.user', 'professional.user', 'service')->get();

        $headers = [
            'Content-Type' => 'text/csv; charset=utf-8',
            'Content-Disposition' => 'attachment; filename=bookings.csv',
        ];

        $callback = function () use ($bookings) {
            $file = fopen('php://output', 'w');
            
            // Headers
            fputcsv($file, [
                'ID', 'Client', 'Professional', 'Service', 'City', 
                'Scheduled At', 'Amount', 'Status', 'Created At'
            ]);

            foreach ($bookings as $booking) {
                fputcsv($file, [
                    $booking->id,
                    $booking->client->user->name,
                    $booking->professional->user->name,
                    $booking->service->name,
                    $booking->city,
                    $booking->scheduled_at,
                    $booking->amount,
                    $booking->status,
                    $booking->created_at,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
