<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Booking;
use App\Models\Transaction;
use App\Models\ProfessionalProfile;
use Illuminate\View\View;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:admin');
    }

    public function index(): View
    {
        $totalClients = User::where('role', 'client')->count();
        $totalProfessionals = User::where('role', 'professional')->count();
        $activeProfessionals = ProfessionalProfile::where('is_published', true)->count();
        $totalBookings = Booking::count();
        $platformRevenue = Transaction::where('status', 'succeeded')->sum('platform_fee');

        $bookingsOverTime = $this->getBookingsOverTime();
        $revenueOverTime = $this->getRevenueOverTime();
        $topCitiesByRevenue = $this->getTopCitiesByRevenue();
        $topServicesByBookings = $this->getTopServicesByBookings();

        return view('admin.dashboard', [
            'totalClients' => $totalClients,
            'totalProfessionals' => $totalProfessionals,
            'activeProfessionals' => $activeProfessionals,
            'totalBookings' => $totalBookings,
            'platformRevenue' => $platformRevenue,
            'bookingsOverTime' => $bookingsOverTime,
            'revenueOverTime' => $revenueOverTime,
            'topCitiesByRevenue' => $topCitiesByRevenue,
            'topServicesByBookings' => $topServicesByBookings,
        ]);
    }

    private function getBookingsOverTime()
    {
        $data = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->toDateString();
            $count = Booking::whereDate('created_at', $date)->count();
            $data[] = ['date' => $date, 'count' => $count];
        }
        return $data;
    }

    private function getRevenueOverTime()
    {
        $data = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->toDateString();
            $revenue = Transaction::whereDate('created_at', $date)
                ->where('status', 'succeeded')
                ->sum('platform_fee');
            $data[] = ['date' => $date, 'revenue' => $revenue];
        }
        return $data;
    }

    private function getTopCitiesByRevenue()
    {
        return Booking::join('transactions', 'bookings.id', '=', 'transactions.booking_id')
            ->where('transactions.status', 'succeeded')
            ->groupBy('bookings.city')
            ->selectRaw('bookings.city, SUM(transactions.platform_fee) as revenue')
            ->orderByRaw('revenue DESC')
            ->limit(10)
            ->get();
    }

    private function getTopServicesByBookings()
    {
        return Booking::join('services', 'bookings.service_id', '=', 'services.id')
            ->where('bookings.status', '!=', 'cancelled')
            ->groupBy('services.name')
            ->selectRaw('services.name, COUNT(*) as count')
            ->orderByRaw('count DESC')
            ->limit(10)
            ->get();
    }
}
