<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\View\View;
use Illuminate\Http\Request;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:admin');
    }

    public function clients(): View
    {
        $clients = User::where('role', 'client')
            ->with('clientBookings')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.users.clients', ['users' => $clients]);
    }

    public function professionals(): View
    {
        $professionals = User::where('role', 'professional')
            ->with('professionalProfile.bookings')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.users.professionals', ['users' => $professionals]);
    }

    public function show($userId): View
    {
        $user = User::findOrFail($userId);

        if ($user->isProfessional()) {
            $user->load('professionalProfile.services', 'professionalProfile.bookings');
        } else {
            $user->load('clientBookings');
        }

        return view('admin.users.show', ['user' => $user]);
    }

    public function updateStatus(Request $request, $userId)
    {
        $user = User::findOrFail($userId);

        $validated = $request->validate([
            'status' => 'required|in:active,suspended,inactive',
        ]);

        $user->update($validated);

        return redirect()->back()->with('success', 'User status updated');
    }

    public function search(Request $request): View
    {
        $query = $request->get('q');
        $role = $request->get('role');

        $users = User::query()
            ->when($role, fn ($q) => $q->where('role', $role))
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                    ->orWhere('email', 'like', "%{$query}%")
                    ->orWhere('city', 'like', "%{$query}%");
            })
            ->paginate(20);

        return view('admin.users.search', ['users' => $users, 'query' => $query]);
    }
}
