<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\Review;
use App\Services\BookingService;
use Illuminate\Http\Request;
use Illuminate\View\View;

class BookingController extends Controller
{
    protected $bookingService;

    public function __construct(BookingService $bookingService)
    {
        $this->bookingService = $bookingService;
        $this->middleware('auth');
    }

    public function create($professionalId): View
    {
        $professional = \App\Models\ProfessionalProfile::with('services')
            ->findOrFail($professionalId);

        return view('booking.create', [
            'professional' => $professional,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'professional_id' => 'required|exists:professional_profiles,id',
            'service_id' => 'required|exists:services,id',
            'scheduled_at' => 'required|date|after:now',
            'notes' => 'nullable|string',
        ]);

        $booking = $this->bookingService->createBooking(
            auth()->id(),
            $validated['professional_id'],
            $validated['service_id'],
            $validated['scheduled_at'],
            $validated['notes'] ?? null
        );

        return redirect()->route('booking.payment', $booking->id);
    }

    public function payment($bookingId): View
    {
        $booking = Booking::with(['client', 'professional', 'service'])
            ->where('client_id', auth()->id())
            ->findOrFail($bookingId);

        return view('booking.payment', [
            'booking' => $booking,
        ]);
    }

    public function confirm(Request $request, $bookingId)
    {
        $booking = Booking::where('client_id', auth()->id())->findOrFail($bookingId);

        $this->bookingService->processPayment($booking, $request->get('payment_intent_id'));

        return redirect()->route('booking.confirmation', $booking->id);
    }

    public function confirmation($bookingId): View
    {
        $booking = Booking::with(['professional', 'service'])
            ->where('client_id', auth()->id())
            ->findOrFail($bookingId);

        return view('booking.confirmation', [
            'booking' => $booking,
        ]);
    }

    public function cancel($bookingId)
    {
        $booking = Booking::where('client_id', auth()->id())->findOrFail($bookingId);
        $this->bookingService->cancelBooking($booking);

        return redirect()->route('client.dashboard')->with('success', 'Booking cancelled');
    }

    public function review($bookingId): View
    {
        $booking = Booking::with(['professional', 'service'])
            ->where('client_id', auth()->id())
            ->findOrFail($bookingId);

        return view('booking.review', ['booking' => $booking]);
    }

    public function submitReview(Request $request, $bookingId)
    {
        $booking = Booking::where('client_id', auth()->id())->findOrFail($bookingId);

        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:500',
        ]);

        Review::create([
            'booking_id' => $booking->id,
            'client_id' => auth()->id(),
            'professional_id' => $booking->professional_id,
            'rating' => $validated['rating'],
            'comment' => $validated['comment'] ?? null,
        ]);

        $booking->professional->updateRating();

        return redirect()->route('client.dashboard')->with('success', 'Review submitted');
    }
}
