<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use Illuminate\View\View;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(): View
    {
        $user = auth()->user();
        
        $upcomingBookings = $user->clientBookings()
            ->where('scheduled_at', '>', now())
            ->where('status', '!=', 'cancelled')
            ->with('professional.user', 'service')
            ->orderBy('scheduled_at')
            ->limit(5)
            ->get();

        $pastBookings = $user->clientBookings()
            ->where('scheduled_at', '<=', now())
            ->where('status', '!=', 'cancelled')
            ->with('professional.user', 'service')
            ->orderBy('scheduled_at', 'desc')
            ->limit(5)
            ->get();

        $favorites = $user->favorites()->limit(6)->get();

        $totalSpent = $user->transactions()
            ->where('status', 'succeeded')
            ->sum('amount');

        return view('client.dashboard', [
            'upcomingBookings' => $upcomingBookings,
            'pastBookings' => $pastBookings,
            'favorites' => $favorites,
            'totalSpent' => $totalSpent,
        ]);
    }

    public function bookings(): View
    {
        $bookings = auth()->user()->clientBookings()
            ->with('professional.user', 'service')
            ->orderBy('scheduled_at', 'desc')
            ->paginate(15);

        return view('client.bookings', ['bookings' => $bookings]);
    }

    public function favorites(): View
    {
        $favorites = auth()->user()->favorites()->paginate(12);

        return view('client.favorites', ['favorites' => $favorites]);
    }

    public function profile(): View
    {
        return view('client.profile');
    }

    public function updateProfile(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'city' => 'nullable|string|max:255',
            'avatar' => 'nullable|image|max:2048',
        ]);

        $user = auth()->user();

        if ($request->hasFile('avatar')) {
            $path = $request->file('avatar')->store('avatars', 'public');
            $validated['avatar'] = asset('storage/' . $path);
        }

        $user->update($validated);

        return redirect()->back()->with('success', 'Profile updated');
    }

    public function paymentHistory(): View
    {
        $transactions = auth()->user()->transactions()
            ->with('booking.professional.user', 'booking.service')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('client.payment-history', ['transactions' => $transactions]);
    }
}
