<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class InstallerController extends Controller
{
    protected $envPath;
    protected $installFlagPath;

    public function __construct()
    {
        $this->envPath = base_path('.env');
        $this->installFlagPath = storage_path('installer_completed.flag');
    }

    /**
     * Check if application is already installed
     */
    public function isInstalled(): bool
    {
        return file_exists($this->installFlagPath);
    }

    /**
     * Redirect if already installed
     */
    public function checkInstallation()
    {
        if ($this->isInstalled()) {
            return redirect('/')->with('info', 'Application is already installed.');
        }
    }

    /**
     * Show installer welcome page
     */
    public function welcome()
    {
        $this->checkInstallation();

        return view('installer.welcome');
    }

    /**
     * Show system requirements check
     */
    public function requirements()
    {
        $this->checkInstallation();

        $requirements = $this->checkSystemRequirements();

        return view('installer.requirements', [
            'requirements' => $requirements,
            'canProceed' => $this->canProceedFromRequirements($requirements),
        ]);
    }

    /**
     * Show database configuration form
     */
    public function database()
    {
        $this->checkInstallation();

        return view('installer.database', [
            'dbDriver' => env('DB_CONNECTION', 'mysql'),
            'dbHost' => env('DB_HOST', 'localhost'),
            'dbPort' => env('DB_PORT', 3306),
            'dbDatabase' => env('DB_DATABASE', 'styleseat'),
            'dbUsername' => env('DB_USERNAME', 'root'),
        ]);
    }

    /**
     * Test database connection
     */
    public function testDatabase(Request $request)
    {
        $validated = $request->validate([
            'driver' => 'required|in:mysql,pgsql,sqlite,sqlsrv',
            'host' => 'required',
            'port' => 'required|numeric',
            'database' => 'required',
            'username' => 'required',
            'password' => 'nullable',
        ]);

        try {
            // Test connection
            DB::purge();
            
            $config = [
                'driver' => $validated['driver'],
                'host' => $validated['host'],
                'port' => $validated['port'],
                'database' => $validated['database'],
                'username' => $validated['username'],
                'password' => $validated['password'] ?? '',
            ];

            config()->set('database.connections.test', $config);
            DB::connection('test')->getPdo();

            // Connection successful
            return response()->json([
                'success' => true,
                'message' => 'Database connection successful!',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Connection failed: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Save database configuration
     */
    public function saveDatabaseConfig(Request $request)
    {
        $this->checkInstallation();

        $validated = $request->validate([
            'driver' => 'required|in:mysql,pgsql,sqlite,sqlsrv',
            'host' => 'required',
            'port' => 'required|numeric',
            'database' => 'required',
            'username' => 'required',
            'password' => 'nullable',
            'create_database' => 'boolean',
        ]);

        try {
            // Create database if requested
            if ($validated['create_database'] && $validated['driver'] === 'mysql') {
                $this->createDatabase($validated);
            }

            // Update .env file
            $this->updateEnvFile([
                'DB_CONNECTION' => $validated['driver'],
                'DB_HOST' => $validated['host'],
                'DB_PORT' => $validated['port'],
                'DB_DATABASE' => $validated['database'],
                'DB_USERNAME' => $validated['username'],
                'DB_PASSWORD' => $validated['password'] ?? '',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Database configuration saved!',
            ]);
        } catch (\Exception $e) {
            Log::error('Database configuration error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving configuration: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Show application configuration form
     */
    public function configuration()
    {
        $this->checkInstallation();

        return view('installer.configuration', [
            'appName' => env('APP_NAME', 'StyleSeat'),
            'appUrl' => env('APP_URL', 'http://localhost:8000'),
            'appEnv' => env('APP_ENV', 'local'),
        ]);
    }

    /**
     * Save application configuration
     */
    public function saveConfiguration(Request $request)
    {
        $this->checkInstallation();

        $validated = $request->validate([
            'app_name' => 'required|string|max:255',
            'app_url' => 'required|url',
            'app_env' => 'required|in:local,production,staging',
            'app_debug' => 'boolean',
        ]);

        try {
            $this->updateEnvFile([
                'APP_NAME' => '"' . $validated['app_name'] . '"',
                'APP_URL' => $validated['app_url'],
                'APP_ENV' => $validated['app_env'],
                'APP_DEBUG' => $validated['app_debug'] ? 'true' : 'false',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Application configuration saved!',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error saving configuration: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Show migration and seeding page
     */
    public function migration()
    {
        $this->checkInstallation();

        return view('installer.migration');
    }

    /**
     * Run migrations
     */
    public function runMigrations()
    {
        try {
            Artisan::call('migrate', ['--force' => true]);

            return response()->json([
                'success' => true,
                'message' => 'Migrations completed successfully!',
                'output' => Artisan::output(),
            ]);
        } catch (\Exception $e) {
            Log::error('Migration error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Migration failed: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Run seeders
     */
    public function runSeeders(Request $request)
    {
        try {
            $seedClass = $request->input('seeder', 'DatabaseSeeder');

            Artisan::call('db:seed', [
                '--class' => $seedClass,
                '--force' => true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Seeding completed successfully!',
                'output' => Artisan::output(),
            ]);
        } catch (\Exception $e) {
            Log::error('Seeding error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Seeding failed: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Show admin user creation form
     */
    public function admin()
    {
        $this->checkInstallation();

        return view('installer.admin');
    }

    /**
     * Create admin user
     */
    public function createAdmin(Request $request)
    {
        $this->checkInstallation();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        try {
            // Use User model to create admin
            $userModel = config('auth.providers.users.model', 'App\Models\User');
            $userModel::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => bcrypt($validated['password']),
                'role' => 'admin',
                'email_verified_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Admin user created successfully!',
            ]);
        } catch (\Exception $e) {
            Log::error('Admin creation error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating admin: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Show completion page
     */
    public function completion()
    {
        $this->checkInstallation();

        return view('installer.completion');
    }

    /**
     * Finalize installation
     */
    public function finalize()
    {
        try {
            // Generate APP_KEY if not set
            if (!env('APP_KEY')) {
                Artisan::call('key:generate', ['--force' => true]);
            }

            // Create installation flag
            File::put($this->installFlagPath, 'installed at ' . now());

            return response()->json([
                'success' => true,
                'message' => 'Installation completed successfully!',
            ]);
        } catch (\Exception $e) {
            Log::error('Finalization error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error finalizing installation: ' . $e->getMessage(),
            ], 422);
        }
    }

    /**
     * Check system requirements
     */
    protected function checkSystemRequirements(): array
    {
        return [
            'php_version' => [
                'name' => 'PHP Version',
                'required' => '8.2+',
                'current' => phpversion(),
                'passed' => version_compare(phpversion(), '8.2', '>='),
            ],
            'extensions' => [
                'name' => 'Required Extensions',
                'required' => 'PDO, OpenSSL, Mbstring, JSON',
                'current' => implode(', ', $this->getLoadedExtensions()),
                'passed' => $this->checkRequiredExtensions(),
            ],
            'storage_writable' => [
                'name' => 'Storage Directory',
                'required' => 'Writable',
                'current' => is_writable(storage_path()) ? 'Writable' : 'Not Writable',
                'passed' => is_writable(storage_path()),
            ],
            'bootstrap_writable' => [
                'name' => 'Bootstrap Cache Directory',
                'required' => 'Writable',
                'current' => is_writable(bootstrap_path('cache')) ? 'Writable' : 'Not Writable',
                'passed' => is_writable(bootstrap_path('cache')),
            ],
            'env_file' => [
                'name' => '.env File',
                'required' => 'Exists and Writable',
                'current' => file_exists($this->envPath) ? 'Exists' : 'Missing',
                'passed' => file_exists($this->envPath) && is_writable($this->envPath),
            ],
        ];
    }

    /**
     * Get loaded extensions
     */
    protected function getLoadedExtensions(): array
    {
        $required = ['PDO', 'openssl', 'mbstring', 'json', 'ctype', 'fileinfo'];
        $loaded = [];

        foreach ($required as $ext) {
            if (extension_loaded($ext)) {
                $loaded[] = $ext;
            }
        }

        return $loaded;
    }

    /**
     * Check if required extensions are loaded
     */
    protected function checkRequiredExtensions(): bool
    {
        $required = ['PDO', 'openssl', 'mbstring', 'json'];

        foreach ($required as $ext) {
            if (!extension_loaded($ext)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if can proceed from requirements
     */
    protected function canProceedFromRequirements(array $requirements): bool
    {
        foreach ($requirements as $req) {
            if (!$req['passed']) {
                return false;
            }
        }

        return true;
    }

    /**
     * Create database
     */
    protected function createDatabase(array $config): bool
    {
        try {
            $connection = new \PDO(
                "mysql:host={$config['host']}:{$config['port']}",
                $config['username'],
                $config['password'] ?? ''
            );

            $connection->exec("CREATE DATABASE IF NOT EXISTS `{$config['database']}`");

            return true;
        } catch (\Exception $e) {
            Log::error('Database creation failed: ' . $e->getMessage());
            throw new \Exception('Could not create database: ' . $e->getMessage());
        }
    }

    /**
     * Update .env file
     */
    protected function updateEnvFile(array $values): void
    {
        $envContent = file_exists($this->envPath)
            ? file_get_contents($this->envPath)
            : '';

        foreach ($values as $key => $value) {
            $pattern = "/^{$key}=.*/m";

            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, "{$key}={$value}", $envContent);
            } else {
                $envContent .= "\n{$key}={$value}";
            }
        }

        if (!file_put_contents($this->envPath, $envContent)) {
            throw new \Exception('Could not write to .env file');
        }

        // Reload environment
        \Dotenv\Dotenv::createImmutable(base_path())->safeLoad();
    }
}
