<?php

namespace App\Http\Controllers\Professional;

use App\Http\Controllers\Controller;
use Illuminate\View\View;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:professional');
    }

    public function index(): View
    {
        $professional = auth()->user()->professionalProfile;
        
        $todayBookings = $professional->bookings()
            ->whereDate('scheduled_at', today())
            ->where('status', '!=', 'cancelled')
            ->with('client.user')
            ->get();

        $upcomingBookings = $professional->bookings()
            ->whereDate('scheduled_at', '>', today())
            ->whereDate('scheduled_at', '<=', today()->addDays(7))
            ->where('status', '!=', 'cancelled')
            ->with('client.user')
            ->orderBy('scheduled_at')
            ->get();

        $weeklyStats = $this->getWeeklyStats($professional);
        $monthlyRevenue = $this->getMonthlyRevenue($professional);

        return view('professional.dashboard', [
            'professional' => $professional,
            'todayBookings' => $todayBookings,
            'upcomingBookings' => $upcomingBookings,
            'weeklyStats' => $weeklyStats,
            'monthlyRevenue' => $monthlyRevenue,
        ]);
    }

    public function bookings(): View
    {
        $professional = auth()->user()->professionalProfile;
        
        $bookings = $professional->bookings()
            ->with('client.user', 'service')
            ->orderBy('scheduled_at', 'desc')
            ->paginate(20);

        return view('professional.bookings', ['bookings' => $bookings]);
    }

    public function calendar(): View
    {
        $professional = auth()->user()->professionalProfile;
        
        return view('professional.calendar', ['professional' => $professional]);
    }

    public function services(): View
    {
        $professional = auth()->user()->professionalProfile;
        $services = $professional->services()->paginate(15);

        return view('professional.services', ['services' => $services]);
    }

    public function editService($serviceId): View
    {
        $professional = auth()->user()->professionalProfile;
        $service = $professional->services()->findOrFail($serviceId);

        return view('professional.edit-service', ['service' => $service]);
    }

    public function updateService(Request $request, $serviceId)
    {
        $professional = auth()->user()->professionalProfile;
        $service = $professional->services()->findOrFail($serviceId);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string',
            'duration' => 'required|integer|min:15',
            'price' => 'required|numeric|min:0',
            'status' => 'required|in:active,inactive',
        ]);

        $service->update($validated);

        return redirect()->route('professional.services')->with('success', 'Service updated');
    }

    public function deleteService($serviceId)
    {
        $professional = auth()->user()->professionalProfile;
        $service = $professional->services()->findOrFail($serviceId);

        $service->delete();

        return redirect()->route('professional.services')->with('success', 'Service deleted');
    }

    public function clients(): View
    {
        $professional = auth()->user()->professionalProfile;
        
        $clients = $professional->bookings()
            ->where('status', '!=', 'cancelled')
            ->distinct()
            ->with('client.user')
            ->get()
            ->groupBy('client_id')
            ->map(function ($bookings) {
                $client = $bookings->first()->client;
                return [
                    'id' => $client->id,
                    'name' => $client->user->name,
                    'email' => $client->user->email,
                    'visits' => count($bookings),
                    'total_spent' => $bookings->sum('amount'),
                    'last_visit' => $bookings->max('scheduled_at'),
                ];
            });

        return view('professional.clients', ['clients' => $clients]);
    }

    public function settings(): View
    {
        $professional = auth()->user()->professionalProfile;

        return view('professional.settings', ['professional' => $professional]);
    }

    public function updateSettings(Request $request)
    {
        $professional = auth()->user()->professionalProfile;

        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'bio' => 'nullable|string|max:1000',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
        ]);

        $professional->update($validated);

        return redirect()->back()->with('success', 'Settings updated');
    }

    private function getWeeklyStats($professional)
    {
        $bookings = $professional->bookings()
            ->whereBetween('scheduled_at', [now()->startOfWeek(), now()->endOfWeek()])
            ->where('status', '!=', 'cancelled')
            ->get();

        return [
            'total_bookings' => count($bookings),
            'completed' => $bookings->where('status', 'completed')->count(),
            'pending' => $bookings->where('status', 'pending')->count(),
        ];
    }

    private function getMonthlyRevenue($professional)
    {
        $days = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->toDateString();
            $revenue = $professional->transactions()
                ->whereDate('created_at', $date)
                ->where('status', 'succeeded')
                ->sum('professional_payout');
            
            $days[] = [
                'date' => $date,
                'revenue' => $revenue,
            ];
        }
        
        return $days;
    }
}
