<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\ProfessionalProfile;
use App\Models\Service;
use App\Models\Booking;
use App\Models\Review;
use App\Models\Transaction;
use App\Models\Availability;
use App\Models\Gallery;
use App\Models\Setting;
use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create admin user
        $admin = User::factory()->admin()->create([
            'name' => 'Admin User',
            'email' => 'admin@styleseat.local',
        ]);

        // Create client users (50-100)
        $clients = User::factory(75)
            ->client()
            ->create();

        // Create professional users (20+)
        $professionals = User::factory(25)
            ->professional()
            ->create();

        // For each professional, create profile, services, and availability
        foreach ($professionals as $professional) {
            $profile = ProfessionalProfile::factory()->create([
                'user_id' => $professional->id,
            ]);

            // Create 3-5 services per professional
            $services = Service::factory(random_int(3, 5))->create([
                'professional_id' => $profile->id,
            ]);

            // Create weekly availability (Monday-Saturday, 9am-6pm)
            for ($day = 1; $day <= 6; $day++) {
                Availability::create([
                    'professional_id' => $profile->id,
                    'day_of_week' => $day,
                    'start_time' => '09:00:00',
                    'end_time' => '18:00:00',
                ]);
            }

            // Create gallery images
            for ($i = 0; $i < 5; $i++) {
                Gallery::create([
                    'professional_id' => $profile->id,
                    'image_url' => 'https://picsum.photos/400/400?random=' . random_int(1, 10000),
                    'alt_text' => 'Portfolio image ' . ($i + 1),
                    'sort_order' => $i,
                ]);
            }

            // Create bookings (10-30 per professional)
            $bookingCount = random_int(10, 30);
            for ($i = 0; $i < $bookingCount; $i++) {
                $client = $clients->random();
                $service = $services->random();
                
                $booking = Booking::factory()->create([
                    'client_id' => $client->id,
                    'professional_id' => $profile->id,
                    'service_id' => $service->id,
                    'city' => $profile->city,
                ]);

                // Create transaction if booking is not cancelled
                if ($booking->status !== 'cancelled') {
                    $amount = $booking->amount;
                    $platformFee = $amount * 0.2;
                    $profitAmount = $amount - $platformFee;

                    Transaction::factory()->succeeded()->create([
                        'booking_id' => $booking->id,
                        'client_id' => $client->id,
                        'professional_id' => $profile->id,
                        'amount' => $amount,
                        'platform_fee' => $platformFee,
                        'professional_payout' => $profitAmount,
                    ]);
                }

                // Create review if booking is completed
                if ($booking->status === 'completed') {
                    Review::factory()->create([
                        'booking_id' => $booking->id,
                        'client_id' => $client->id,
                        'professional_id' => $profile->id,
                    ]);
                }
            }

            // Update professional stats
            $profile->update([
                'total_bookings' => $profile->bookings()->where('status', '!=', 'cancelled')->count(),
                'total_revenue' => $profile->transactions()->where('status', 'succeeded')->sum('professional_payout'),
            ]);
            
            $profile->updateRating();
        }

        // Seed initial settings
        $this->seedSettings();
    }

    private function seedSettings(): void
    {
        $settings = [
            ['key' => 'app_name', 'value' => 'StyleSeat', 'type' => 'string', 'group' => 'app'],
            ['key' => 'app_logo_url', 'value' => 'https://via.placeholder.com/200x50?text=StyleSeat', 'type' => 'string', 'group' => 'app'],
            ['key' => 'primary_color', 'value' => '#6366F1', 'type' => 'string', 'group' => 'app'],
            ['key' => 'secondary_color', 'value' => '#8B5CF6', 'type' => 'string', 'group' => 'app'],
            
            ['key' => 'stripe_mode', 'value' => 'test', 'type' => 'string', 'group' => 'stripe'],
            ['key' => 'platform_commission_percentage', 'value' => '20', 'type' => 'integer', 'group' => 'stripe'],
            
            ['key' => 'firebase_project_id', 'value' => env('FIREBASE_PROJECT_ID'), 'type' => 'string', 'group' => 'firebase'],
            
            ['key' => 'mail_from_address', 'value' => 'noreply@styleseat.local', 'type' => 'string', 'group' => 'email'],
            ['key' => 'mail_from_name', 'value' => 'StyleSeat', 'type' => 'string', 'group' => 'email'],
            
            ['key' => 'booking_confirmation_required', 'value' => '1', 'type' => 'boolean', 'group' => 'booking'],
            ['key' => 'cancellation_window_hours', 'value' => '24', 'type' => 'integer', 'group' => 'booking'],
        ];

        foreach ($settings as $setting) {
            Setting::firstOrCreate(
                ['key' => $setting['key']],
                $setting
            );
        }
    }
}
